#include "CGen2SimpleTagDialog.h"

#include <QMessageBox>

#include <QrfeTagManipulatorInterface>
#include <reader/QrfeRfePurReader>
#include <tag/QrfeGen2Tag>

#include "sub/CGen2KillDialog.h"
#include "sub/CGen2LockDialog.h"

CGen2SimpleTagDialog::CGen2SimpleTagDialog(QWidget *parent)
    : QDialog(parent)
{
	ui.setupUi(this);
	setInfoEdit(ui.infoEdit);

	connect(ui.okButton, 			SIGNAL(clicked()), 					this, SLOT(accept()));
	connect(ui.readButton, 			SIGNAL(clicked()), 					this, SLOT(read()));
	connect(ui.writeButton, 		SIGNAL(clicked()), 					this, SLOT(write()));
	connect(ui.lockButton, 			SIGNAL(clicked()), 					this, SLOT(lock()));
	connect(ui.killButton, 			SIGNAL(clicked()), 					this, SLOT(kill()));

	connect(ui.memBankBox, 			SIGNAL(currentIndexChanged(int)), 	this, SLOT(paramsChanged()));
	connect(ui.wordAddressBox, 		SIGNAL(valueChanged(int)), 			this, SLOT(paramsChanged()));
	connect(ui.wordCountBox, 		SIGNAL(valueChanged(int)), 			this, SLOT(paramsChanged()));

	m_killDialog 			= new CGen2KillDialog(this);
	m_lockDialog 			= new CGen2LockDialog(this);
}

CGen2SimpleTagDialog::~CGen2SimpleTagDialog()
{
	delete m_killDialog;
	delete m_lockDialog;
}


int CGen2SimpleTagDialog::exec(QrfeReaderInterface* ph, QString tagId, uint antennaId)
{
	m_reader = ph;
	m_antennaId = antennaId;

	if(m_reader->tagType() != QrfeGlobal::TAG_GEN2)
		return 0;

	QrfeTagManipulatorInterface* 	tagManipulator;
	if((tagManipulator = m_reader->getTagManipulator()) == 0 || dynamic_cast<QrfeGen2ManipulatorInterface*>(tagManipulator) == 0)
		return 0;

	m_tag = new QrfeGen2Tag(tagId, dynamic_cast<QrfeGen2ManipulatorInterface*>(tagManipulator), this);

	if(m_antennaId != 0)
	{
		m_reader->setWorkingAntenna(antennaId);
	}

	clearInfo();

	ui.tagIdEdit->setText(m_tag->tagId());

	ui.memBankBox->setCurrentIndex(0);
	ui.wordAddressBox->setValue(0);
	ui.wordCountBox->setValue(1);
	ui.accessPasswordEdit->setText("00-00-00-00");

	ui.dataLineEdit->setInputMask("");
	ui.dataLineEdit->setText("");

	ui.writeButton->setEnabled(false);

	return QDialog::exec();
}

void CGen2SimpleTagDialog::done(int r)
{
	delete m_tag;

	QDialog::done(r);
}

void CGen2SimpleTagDialog::read()
{
	clearInfo();

	if(!ui.accessPasswordEdit->hasAcceptableInput())
	{
		QMessageBox::critical(this, "Error", "The password is not valid!");
		return;
	}

	bool ok = false;
	QByteArray passw = QrfeGlobal::stringToBytes(ui.accessPasswordEdit->text(), &ok);
	if(!ok)
	{
		QMessageBox::critical(this, "Error", "The password is not valid!");
		return;
	}

	uchar 	membank = ui.memBankBox->currentIndex();
	ushort 	address = ui.wordAddressBox->value();
	uchar 	count   = ui.wordCountBox->value() * 2;

	QrfeGlobal::TMIResult res = QrfeGlobal::TMI_ERROR;

	QByteArray data;

	res = m_tag->readFromTag((QrfeGen2Tag::MEMORY_BANK)membank, address, passw, count, data);

	QString inputMask;
	QString text;
	if(res == QrfeGlobal::TMI_OK)
	{
		infoSetOK("-- Read data from tag - OK -- ");
		infoAppend("Read the data: " + QrfeGlobal::bytesToString(data));

		for(int i = 0; i < data.size(); i++)
			inputMask += "HH-";
		inputMask.remove(inputMask.size()-1, 1);

		text = QrfeGlobal::bytesToString(data);

		ui.writeButton->setEnabled(true);
	}
	else
	{
		inputMask = "";
		text = "";
		handleError(res, "Could not read data from the tag");
	}

	ui.dataLineEdit->setInputMask(inputMask);
	ui.dataLineEdit->setText(text);
}

void CGen2SimpleTagDialog::write()
{
	bool ok;

	clearInfo();

	if(!ui.accessPasswordEdit->hasAcceptableInput())
	{
		QMessageBox::critical(this, "Error", "The password is not valid!");
		return;
	}

	ok = false;
	QByteArray passw = QrfeGlobal::stringToBytes(ui.accessPasswordEdit->text(), &ok);
	if(!ok)
	{
		QMessageBox::critical(this, "Error", "The password is not valid!");
		return;
	}

	if(!ui.dataLineEdit->hasAcceptableInput())
	{
		QMessageBox::critical(this, "Error", "The data is not valid!");
		return;
	}

	ok = false;
	QByteArray data = QrfeGlobal::stringToBytes(ui.dataLineEdit->text(), &ok);
	if(!ok)
	{
		QMessageBox::critical(this, "Error", "The data is not valid!");
		return;
	}

	uchar 	membank = ui.memBankBox->currentIndex();
	ushort 	address = ui.wordAddressBox->value();

	QrfeGlobal::TMIResult res = QrfeGlobal::TMI_ERROR;

	res = m_tag->writeToTag((QrfeGen2Tag::MEMORY_BANK)membank, address, passw, data);

	if(res == QrfeGlobal::TMI_OK)
	{
		infoSetOK("-- Wrote data from tag - OK -- ");
		infoAppend("Wrote the data: " + QrfeGlobal::bytesToString(data));
	}
	else
	{
		handleError(res, "Could not write data from the tag");
	}
}

void CGen2SimpleTagDialog::lock()
{
	m_lockDialog->exec(m_tag);
}

void CGen2SimpleTagDialog::kill()
{
	m_killDialog->exec(m_tag);
}


void CGen2SimpleTagDialog::paramsChanged()
{
	ui.writeButton->setEnabled(false);
}

